/*****************************************************************************

MultiKitB: AVR Oscilloscope and Development Kit

Gabotronics C.A.
February 2009

Copyright 2009 Gabriel Anzziani

This program is distributed under the terms of the GNU General Public License 

www.gabotronics.com
email me at: gabriel@gabotronics.com

AVR DEVELOPMENT KIT:
1) Oscilloscope: Fully featured with FFT
2) SKTCH: Popular painting game
3) 3D on my LCD: 3D demo using rotary encoders to change viewing angle
4) Mini Piano: Mini piano demo
5) Fractal: Fractal to demo simple math and random numbers
6) RS232: RS-232 Terminal

AVR Fuse settings:
BODLEVEL    4.3V
OCDEN       OFF
JTAGEN      OFF
SPIEN       ON
WDTON       OFF
EESAVE      ON
BOOTSZ      128
BOOTRST     OFF
CKDIV       OFF
CKOUT       OFF
SUT_CKSEL   EXT CRYST 8MHz 16K CK + 65ms (Crystal is 19.2MHz)

Compiled with GCC, -O2 optimizations

*****************************************************************************/

#include <avr/io.h>
#include <util/delay_basic.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include "mygccdef.h"
#include "sed1335.h"
#include "multikit.h"
#include "asmutil.h"

// Fuse settings if using ELF file
FUSES = {
    .low = 0xFF,                                    // EXT CRYST 8MHz 16K CK + 65ms
    .high = (FUSE_SPIEN & FUSE_EESAVE),             // SPI enabled, Save EE
    .extended = (FUSE_BODLEVEL1 & FUSE_BODLEVEL0),  // BODLEVEL    4.3V
};

void DEMO(void);
void BALL(void);
void THREED(void);
void PIANO(void);
void FRACTAL(void);
void TERMINAL(void);

// C 8-bit Half Sine Table
signed char EEMEM sint[128] = {
   0,   3,   6,   9,  12,  16,  19,  22,  25,  28,  31,  34,  37,  40,  43,  46,
  49,  51,  54,  57,  60,  63,  65,  68,  71,  73,  76,  78,  81,  83,  85,  88,
  90,  92,  94,  96,  98, 100, 102, 104, 106, 107, 109, 111, 112, 113, 115, 116,
 117, 118, 120, 121, 122, 122, 123, 124, 125, 125, 126, 126, 126, 127, 127, 127,
 127, 127, 127, 127, 126, 126, 126, 125, 125, 124, 123, 122, 122, 121, 120, 118,
 117, 116, 115, 113, 112, 111, 109, 107, 106, 104, 102, 100,  98,  96,  94,  92,
  90,  88,  85,  83,  81,  78,  76,  73,  71,  68,  65,  63,  60,  57,  54,  51,
  49,  46,  43,  40,  37,  34,  31,  28,  25,  22,  19,  16,  12,   9,   6,   3 };
 
// Return sine fractional value
signed char Sin(byte angle) {
    if(angle>=128) return -(signed char)eeprom_read_byte(&sint[angle-128]);
    return (signed char)eeprom_read_byte(&sint[angle]);
}

// Return cosine fractional value
signed char Cos(byte angle) {
    return Sin(angle+64);
}

 // Keypad
byte keylevel[16] PROGMEM =
{ 247, 233, 221, 209, 199, 190, 178, 166, 160, 151, 136, 125, 122, 116, 107, 51 };

byte demowave(byte i, byte k);

// Global variables
volatile byte key=0,oldkey=0;    // Keypad
volatile byte ROT1=0,ROT2=0,MAX1=255,MAX2=255;    // Encoder handler variables

// linear congruential generator
byte random(void) {
    static unsigned int Seed = 0;
    Seed = 25173 * Seed + 1;
    return hibyte(Seed);
}

void sound(byte tone) {
    byte i,j;
    for(j=80; j; j--) {
        for(i=tone; i; i--) {
            asm("nop");
            asm("nop");
        }
		setbit(PIND, BUZZER);   // Toggle pin
	}
	clrbit(PORTD, BUZZER);
}

// Negative voltage module control
void Negative(byte set) {
    TCCR2A = 0x00;              // stop
    TCCR2B = 0x00;
    ASSR  = 0x00;               // set async mode
    if(set) {                   // Enable negative voltage
        clrbit(PRR,PRTIM2);
        TCCR2A = 0x42;          // start, OC2A toggle, waveform = CTC
        TCCR2B = 0x01;          // clk/1, freq
        OCR2A = 2;
    }                           // Disable negative voltage
    else {
        setbit(PRR,PRTIM2);     // power down tmr2
    }
}

int main(void) {
    byte i,j=0,data;
    delay_ms(100);  // Debounce ON Switch
	DIDR0 = 0x01;   // Disable digital input buffers
    PORTA = 0x10;   // Disable ADS931
	DDRA =  0xFC;   // SEL0-SEL2 outputs, PA2 output, LED output
    PORTB = 0x08;   // Disable ADS931 output
	DDRB =  0x1F;   // PB5-PB7 inputs
    ACSR =  0x80;   // Disable comparator
	PORTC = 0xFF;
	DDRC =  0xFF;   // Data port - LCD and ADC
	PORTD = 0x3C;   // Pull ups on rotary encoders
	DDRD =  0xC2;
    ADMUX = 0x20;   // ADC left adjusted, select CH0
    ADCSRA= 0x86;   // Enable ADC, XTAL/64 frequency
    PCICR = 0x09;   // Enable interrupt on change for PCIE0 and PCIE3
    PCMSK3= 0x3C;   // Rotary encoder pins change interrupt mask
    PCMSK0= 0x02;   // Power button switch pin change interrupt mask

    WDTCSR |= (1<<WDCE) | (1<<WDE); // Enable change on watchdog settings
    WDTCSR = (1<<WDIE) | (1<<WDP1)/* | (1<<WDP0)*/;   // Enable watchdog interrupt with 125mS period
    
    setbit(PIND, BUZZER);   // Toggle pin
    // Initialize LCD
    lcd_init(TEXT);
    lcd_puts(PSTR("MULTIKITB V1.2"));
    setbit(PIND, BUZZER);   // Toggle pin

    while(testbit(PINA, ONBUTTON)) PORTA = 0x10; // Wait until switch is depressed
    PORTA = 0x14;                   //Keep power up with PA2
    delay_ms(20);
    PCIFR = 0x0F;   // Clear PCIE flags (bouncing on ON button)

    sei();          // enable global interrupts
    setbit(Status, update);

    for(;;) {
        SMCR = 0x00;    // Sleep mode = idle, sleep disabled
        CLKPR = 0x80;   // Change clock prescaler
        CLKPR = 0x00;   // Clock prescaler = 1 (Full AVR Speed)
        PRR = 0xFF;     // Power down all AVR peripherals
        if(testbit(Status, update)) {
            clrbit(Status, update);
            lcd_init(TEXT);
            Negative(0);                // Turn off Negative Voltage
            TCCR0A = 0;
            TCCR0B = 0;                 // Stop ADS931 clock
            PORTA = 0x14;               // Keep power up with PA2, disable ADS931

            lcd_goto(3,1); lcd_puts(PSTR("Gabotronics"));
            lcd_goto(7,3); lcd_puts(PSTR("AVR"));
            lcd_goto(4,4); lcd_puts(PSTR("MultiKitB"));
            // Curson direction  
            lcd_write_command(CSR_DOWN);
            lcd_goto(0,12); lcd_puts(PSTR("DEMO"));
            lcd_goto(3,11); lcd_puts(PSTR("SKTCH"));
            lcd_goto(6,14); lcd_puts(PSTR("3D"));
            lcd_goto(9,11); lcd_puts(PSTR("PIANO"));
            lcd_goto(12,11); lcd_puts(PSTR("FRCTL"));
            lcd_goto(15,11); lcd_puts(PSTR("RS232"));
        }
        for(i=0; i<128; i++) {  // Intro waveform
            data = demowave(i,j+1); pixel(i,data>>1,0); // erase old
            data = demowave(i,j);   pixel(i,data>>1,255); // plot  new
        }
        j--;
        setbit(SMCR,SE);      // Enable Sleep
        sleep_cpu();
        if(!key && oldkey) {  // detect key release
            switch(oldkey) {
                case K1: DEMO(); break;
                case K2: SKTCH(); break;
                case K3: THREED(); break;
                case K4: PIANO(); break;
                case K5: FRACTAL(); break;
                case K6: TERMINAL(); break;
            }
            oldkey=0;
            key = 0;
            UCSR0A = 0; UCSR0B = 0; UCSR0C = 0; // Disable UART
            TCCR2A = 0; TCCR2B = 0;             // Disable Timer2
            setbit(Status, update);
        }
    }
    return 0;
}

byte demowave(byte index, byte k) {
    return 128 + fmuls_8(Sin(index)>>1, Sin((byte)((index<<1)+k)) );
}

// Watchdog interrupt, 64mS period
// This interrupt is used among all applications for user input
ISR(WDT_vect) {
    static byte oldin=0;
    byte i;
    clrbit(PRR,PRADC);              // Power up ADC
    setbit(ADCSRA, ADSC);	        // Start conversion
    clrbit(SMCR,SE);                // Disable Sleep
	while(testbit(ADCSRA,ADSC));    // wait
	// Read Buttons
	if(oldin!=ADCH) {
		for(i=16; i>0; i--) {			    // DeBounce
       		setbit(ADCSRA, ADSC);		    // Start conversion (key input)
       		while(testbit(ADCSRA,ADSC));    // wait
			if(oldin!=ADCH) { oldin=ADCH; i++; }
            oldkey = key;
            setbit(Status, update);         // Valid key
		}
		for(key=0; key<16; key++) {
            if(ADCH>(byte)pgm_read_byte_near(keylevel+key)) break;
        }
    }
    setbit(PRR,PRADC);  // Power down ADC
    if(testbit(UCSR0A,RXC0)) {
       i=UDR0;
       if(i=='C') SendBMP();    // XMODEM CRC file request
    }
}

// POWER button
ISR(PCINT0_vect) {
    byte i=0,j;
    delay_ms(10);
    oldkey = key;
    if(testbit(PINA, ONBUTTON)) {
        setbit(Status, update);         // Valid key
        key = K7;
    }
    while(testbit(PINA, ONBUTTON)) {
        delay_ms(10);
        i++;
        if(i==125) {
            cli();  // disable interrupts
            // Shut down sound
            for(i=0; i<40; i++) {
		        for(j=i; j; j--) {
                    delay_ms(1);
                }
		        setbit(PIND, BUZZER);   // Toggle pin
	        }
            for(;;) PORTA = 0x10;  // TURN OFF SYSTEM!!!
        }
    }
}

// Rotary Encoders
//                _______    
// OutA:  _______|       |_____
//            _______     
// OutB: ____|       |______
//                      
// Data:   0   1   3   2
//
ISR(PCINT3_vect) {
    static byte oldEnc1=0,oldEnc2=0, Decision1=0, Decision2=0;
    byte Enc1, Enc2, Go1=0, Go2=0;

    Enc1 = (PIND & 0x0C) >> 2;
    Enc2 = (PIND & 0x30) >> 4;
    // Check Encoder 1
    if((Enc1==0) && (Decision1==0)) {  // Encoder is 00
        if(oldEnc1==1) Go1=2; // Increment
        if(oldEnc1==2) Go1=1; // Decrement
        Decision1=1;
    }
    else if((Enc1==3) && (Decision1==1)) {  // Encoder is 11
        if(oldEnc1==2) Go1=2; // Increment
        if(oldEnc1==1) Go1=1; // Decrement
        Decision1=0;
    }
    // Check Encoder 2
    if((Enc2==0) && (Decision2==0)) {  // Encoder is 00
        if(oldEnc2==1) Go2=2; // Increment
        if(oldEnc2==2) Go2=1; // Decrement
        Decision2=1;
    }
    else if((Enc2==3) && (Decision2==1)) {  // Encoder is 11
        if(oldEnc2==2) Go2=2; // Increment
        if(oldEnc2==1) Go2=1; // Decrement
        Decision2=0;
    }

    oldEnc1 = Enc1;
    oldEnc2 = Enc2;

    if(Go1==1) {
        if(ROT1<MAX1) { ROT1++; sound(50); }
        else { sound(250); ROT1=0; }  // Roll over
    }
    if(Go1==2) {
        if(ROT1>0) { ROT1--; sound(50); }
        else { sound(250); ROT1=MAX1; } // Roll over
    }

    if(Go2==1) {
        if(ROT2<MAX2) { ROT2++; sound(50); }
        else { sound(250); ROT2=0; } // Roll over
    }
    if(Go2==2) {
        if(ROT2>0) { ROT2--; sound(50); }
        else { sound(250); ROT2=MAX2; } // Roll over
    }

    if(Go1 || Go2) setbit(Status, update);
}
