/*****************************************************************************

MultiKitB: AVR Oscilloscope and Development Kit

Gabotronics C.A.
February 2009

Copyright 2009 Gabriel Anzziani

This program is distributed under the terms of the GNU General Public License 

www.gabotronics.com
email me at: gabriel@gabotronics.com

The Sierpinski Triangle and Hexagon Fractals
The random function is cyclical, if you watch the fractal long enough,
it will clear all the dots for a moment!

The Mandelbrot Fractal uses Fractional numbers to reduce code size and speed
up execution, however, it will limit the zoom in.

*****************************************************************************/

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <util/delay_basic.h>
#include <avr/sleep.h>
#include "mygccdef.h"
#include "multikit.h"
#include "sed1335.h"
#include "asmutil.h"

void mandelbrot(fixed xx0, fixed yy0, fixed w);
void Koch(byte depth, byte level,fixed x1, fixed y1, fixed x2, fixed y2 );

void FRACTAL(void) {
    byte fractal=0;
    fixed mx=float2fix(-1.5),my=float2fix(-1),mw=float2fix(0.015625);
    lcd_init(TEXT);
    lcd_write_command(CSR_DOWN); lcd_puts(PSTR("Fractal"));
    CLKPR = 0x80;       // Change clock prescaler
    CLKPR = 0x04;       // Clock prescaler = 16
    signed int x=0, y=-64;
    byte next;
    for(;;) {
        next = random();
        switch(fractal) {
            case 0: // Triangle
                pixel(x+64,y+64,128);   // Toggle pixel
                if(next<85)         { x=x/2;      y=(y-64)/2; }   // (0,-64)
                else if(next<170)   { x=(x-64)/2; y=(y+63)/2; }  // (-64,63)
                else                { x=(x+63)/2; y=(y+63)/2; }  // (63,63)
            break;
            case 1: // Hexagon
                pixel(x+64,y+64,128);   // Toggle pixel
                if(next<51)         { x=(-x)/3;      y=(-128-y)/3; }     // (0,-64)
                else if(next<102)   { x=(-128-x)/3;  y=(-32-y)/3; }  // (-64,-16)
                else if(next<153)   { x=(127-x)/3;  y=(-32-y)/3; }  // (63,-16)
                else if(next<204)   { x=(-80-x)/3;   y=(127-y)/3; }  // (-32,63)
                else                { x=(80-x)/3;   y=(127-y)/3; }  // (32,63)
            break;
           // case 2: mandelbrot(mx,my,mw); break;
        }
        // Check key input
        if(testbit(Status, update) && key) {
            clrbit(Status, update);
            lcd_clear_graphics();
            switch(key) {
                case K1: fractal=0; CLKPR = 0x80; CLKPR = 0x04; break;
                case K2: fractal=1; CLKPR = 0x80; CLKPR = 0x04; break;
                case K3: fractal=2; CLKPR = 0x80; CLKPR = 0x00; break;
                case K4: fractal=3; CLKPR = 0x80; CLKPR = 0x00; break;
                case K5: mw++; mx-=16*mw; my-=16*mw; break;
                case K6: if(mw>1) { mx+=16*mw; my+=16*mw; mw--; } break;
                case K7: return;
            }
            while(key) delay_ms(50);
            if(fractal==2) kochdemo();
            if(fractal==3) mandelbrot(mx,my,mw);
        }
    }
}

void mandelbrot(fixed xx0, fixed yy0, fixed w) {
    fixed x2,y2,x0,y0,d,x,y;
    unsigned char i, j, k;
    d = w; // calculate increments
    // repeat on each screen pixel
    y0 = yy0;
    for (i=0; i<128; i++) {
        x0 = xx0;
        for (j=0; j<128; j++) {
            // initialization
            x = x0;
            y = y0;
            k = 0;
            do { // core iteration
                x2 = fixmult(x,x);
                y2 = fixmult(y,y);
                y = fixmult(x,y);
                y = fixmult(float2fix(2),y);

                y = y + y0;
                x = x2 - y2 + x0;
                k++;
            } while ( (x2 + y2 < float2fix(4))  && ( k < 64));
            // check if the point belongs to the Mandelbrot set
            if ( k == 64) pixel(j,i,255);
            else if ( k & 2) pixel(j,i,k);   // draw bands
            x0 += d;    // compute next point x0
        }
        y0 += d;       // compute next y0
    }
}

void Koch (byte depth, byte level,fixed x1, fixed y1, fixed x2, fixed y2 ) {
    static fixed cosa = float2fix(0.5);
    static fixed sina = float2fix(-0.8660254); //-sqrt(3.0)/2.0;
	if (level == depth) {
		lcd_line ( fix2int(x1), fix2int(y1)-64, fix2int(x2), fix2int(y2)-64, 255 );
	}
	else {
		fixed dx = divfix(x2 - x1,float2fix(3)); 
		fixed dy = divfix(y2 - y1,float2fix(3));
		fixed xa = x1 + dx;
		fixed ya = y1 + dy;
		fixed xb = x2 - dx;
		fixed yb = y2 - dy;
		fixed xc = xa + fixmult(dx,cosa) - fixmult(dy,sina);
		fixed yc = ya + fixmult(dy,cosa) + fixmult(dx,sina);
		Koch ( depth+1, level, x1, y1, xa, ya );
		Koch ( depth+1, level, xa, ya, xc, yc );
		Koch ( depth+1, level, xc, yc, xb, yb );
		Koch ( depth+1, level, xb, yb, x2, y2 );
	}
}

void kochdemo() {
	for (int level=0; level<7; level++ ) {
        lcd_clear_graphics();
		Koch (0, level, float2fix(0), float2fix(128), float2fix(128), float2fix(128));
		delay_ms(250);
		delay_ms(250);
	}
}

