#include <avr/io.h>
#include <avr/pgmspace.h>
#include <util/delay_basic.h>
#include <avr/sleep.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "mygccdef.h"
#include "multikit.h"
#include "sed1335.h"
#include "asmutil.h"

byte end=0;

// Linked List implementation

// Particle definition
typedef struct {
    fixed x,y;      // Current position
    byte ox,oy;     // Old position
    fixed vx,vy;    // Speed
//    byte q;         // Charge
//    byte w;         // Weight
} particle;

// First in - First out stack
struct list {
    struct list *next;
    particle p;
};

typedef struct list item;
typedef item* Queue;

// Create new empty item
item *newitem(void) { 
    item *pointer;
    pointer = (item *)malloc(sizeof(item));
    if(pointer==NULL) {
        return NULL;
    }
    pointer->next=NULL;
    pointer->p.x=float2fix(32);  pointer->p.y=float2fix(32);
    pointer->p.ox=32; pointer->p.oy=32;
    pointer->p.vx=float2fix(0);  pointer->p.vy=float2fix(0);
//    pointer->p.q=1; pointer->p.w=1;
    return pointer;
}

// Adds item I to the end of the stack
void Enqueue(Queue *Q, item *I) {
    item *p;
    I->next=NULL;
    if(*Q==NULL) {
        *Q=I;
    }
    else {
        p=*Q;
        while(p->next!=NULL) p=p->next;
        p->next=I;
    }
}

// Remove item at top of stack
item *Firstout(Queue *Q) {
    item *p;
    p = *Q;
    if(*Q!=NULL) {
        *Q=(*Q)->next;
        p->next=NULL;
    }
    return p;
}

// Remove item at the end of the stack
item *Lastout(Queue *Q) {
    item *p,*o;
    if(*Q == NULL) return NULL; // empty list
    p = *Q;
    while((p->next->next)!=NULL) p=p->next;
    o=p->next;
    p->next=NULL;
    return o;
}

// Generate new particle
item *create(Queue *Q) {
    item *t;
    t = newitem();
    if (t!=NULL) {
        t->p.x = int2fix(random()/2);
        t->p.y = int2fix(random()/2);
        Enqueue(Q, t);
    }
    return t;
}

// Particle demo!!!
void DEMO(void) {
    Queue Q;        // List of particles
    item *t,*r;     // Pointers to a particle
    fixed f,d,dx,dy,cos,sin,sdist,temp;
    byte sx,sy,n=1,i;
    end=0;

    // Timer 2 will control the refresh rate
    clrbit(PRR,PRTIM2); // Power up Timer 2
    TCCR2A = 2;     // Clear Timer on Compare mode
    TCCR2B = 7;     // clk/1024
    OCR2A = 255;    // --> Timer 2 period = 73Hz

    lcd_init(TEXT);
    setbit(Status, friction);
    setbit(Status, gravity);
    setbit(Status, forcetype);  // spring
    setbit(Status, chain);
    Q=newitem();    // create first particle
    for(i=32; i<64; i+=2) {
        t = newitem();
        t->p.x = int2fix(i);
        t->p.y = int2fix(16);
        Enqueue(&Q, t);
        n++;
    }
    ROT1 = 32; MAX1 = 127;
    ROT2 = 16; MAX2 = 127;
    setbit(Status, update);
    while(!end) {
        Q->p.x = ROT1<<8;   // Encoders control the position of the first particle
        Q->p.y = ROT2<<8;
        // Check key input
        if(testbit(Status, update)) {
            clrbit(Status, update);
            switch(key) {
                case K1: togglebit(Status, friction); break;
                case K2: togglebit(Status, gravity); break;
                case K3: togglebit(Status, chain); break;
                case K4: togglebit(Status, forcetype); break;
                case K5: if(Q!=NULL) {
                            t=Lastout(&Q); n--;
                         }
                         pixel(t->p.ox,t->p.oy,0);
                         free(t);
                break;
                case K6: if(create(&Q)!=NULL) n++; break;
                case K7: end=1; key=0; break;
            }
            lcd_goto(0,15); lcd_puts(PSTR("                "));
            if(testbit(Status, friction)) { lcd_goto(0,15); lcd_puts(PSTR("FR")); }
            if(testbit(Status, gravity)) { lcd_goto(3,15); lcd_puts(PSTR("GR")); }
            if(testbit(Status, chain)) { lcd_goto(6,15); lcd_puts(PSTR("CH")); }
            lcd_goto(9,15);
            if(testbit(Status, forcetype)) lcd_puts(PSTR("SP"));
            else lcd_puts(PSTR("EL"));
            lcd_goto(12,15); PrintN(n);
            while(key) delay_ms(50);
        } else while(!testbit(TIFR2,OCF2A)) ;   // Sync - 73Hz
        setbit(TIFR2,OCF2A);    // clear bit

        // process all particles
        t=Q;
        while(t!=NULL) {
            // Friction
            if(testbit(Status, friction)) {
                // Viscocity
                // Multiply v by 255, which is around 0.995 in fixed format
                if(t->p.vx>0) t->p.vx = multfix(t->p.vx,255);
                if(t->p.vx<0) t->p.vx = -1*multfix(-(t->p.vx),255);
                if(t->p.vy>0) t->p.vy = multfix(t->p.vy,255);
                if(t->p.vy<0) t->p.vy = -1*multfix(-(t->p.vy),255);
            }
            // Tweak to reduce instability
/*            i++;
            if(i=16) {
                i=0;
                if(t->p.vx>0) t->p.vx--;
                if(t->p.vx<0) t->p.vx++;
                if(t->p.vy>0) t->p.vy--;
                if(t->p.vy<0) t->p.vy++;
            }*/
            // Gravity
            if(testbit(Status, gravity)) {
                t->p.vy+=2;
            }
            // Limit speed
            if(t->p.vx>float2fix(5)) t->p.vx = float2fix(5);
            if(t->p.vx<float2fix(-5)) t->p.vx = float2fix(-5);
            if(t->p.vy>float2fix(5)) t->p.vy = float2fix(5);
            if(t->p.vy<float2fix(-5)) t->p.vy = float2fix(-5);
            // Move
            t->p.x=t->p.x+t->p.vx;  // speed x
            t->p.y=t->p.y+t->p.vy;  // speed y
            // Check boundaries
            if(t->p.x<float2fix(0))   { t->p.x=float2fix(0); t->p.vx = abs(t->p.vx); }
            if(t->p.x>float2fix(127)) { t->p.x=float2fix(127); t->p.vx = -abs(t->p.vx); }
            if(t->p.y<float2fix(0))   { t->p.y=float2fix(0); t->p.vy = abs(t->p.vy); }
            if(t->p.y>float2fix(127)) { t->p.y=float2fix(127); t->p.vy = -abs(t->p.vy); }
            // Force between all other particles
            r=t->next;
            while(r!=NULL) {
                if(t->p.x>r->p.x) {sx=0; dx = (t->p.x-r->p.x); }      // delta x
                else { sx=1;  dx = (r->p.x-t->p.x); }
                sdist  = hibyte(dx)*hibyte(dx);
                if(t->p.y>r->p.y) {sy=0; dy = (t->p.y-r->p.y); }      // delta x
                else { sy=1;  dy = (r->p.y-t->p.y); }
                sdist += hibyte(dy)*hibyte(dy);    // sdist = squared of distance
                d = sqrt(sdist);           // distance
                if(d>2) {
                    cos = dx/d;
                    sin = dy/d;
                    temp = 1;//t->p.q*r->p.q;
                    if(testbit(Status, forcetype)) { // spring
                        f=4*d;   // F= k*x
                    }
                    else {      // electric charge
                        f = divfix(temp,sdist);     // F = q1*q2/r^2
                    }
                    if(sx) { t->p.vx+= multfix(f,cos); r->p.vx-= multfix(f,cos); }
                    else   { t->p.vx-= multfix(f,cos); r->p.vx+= multfix(f,cos); }
                    if(sy) { t->p.vy+= multfix(f,sin); r->p.vy-= multfix(f,sin); }
                    else   { t->p.vy-= multfix(f,sin); r->p.vy+= multfix(f,sin); }
                }
                if (testbit(Status, chain)) break;  // Only check force with once
                else r=r->next;
            }
            // draw particles
            pixel(t->p.ox,t->p.oy,0);
            pixel(t->p.ox=fix2int(t->p.x),t->p.oy=fix2int(t->p.y),255);
            t=t->next;
        }
    }
    // free RAM
    while(Q!=NULL) {
        t = Q;
        Q = Q->next;
        free(t);        // delete item
    }
}
