/***********************************************************************

MultiKitB: AVR Oscilloscope and Development Kit

Gabotronics C.A.
February 2009

Copyright 2009 Gabriel Anzziani

This program is distributed under the terms of the GNU General Public License 

www.gabotronics.com
email me at: gabriel@gabotronics.com

Simple RS-232 application and functions
    19200 bps
    8 data bits
    1 stop bit
    no parity

*****************************************************************************/

#include <avr/io.h>
#include <util/delay_basic.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/eeprom.h>
#include "mygccdef.h"
#include "sed1335.h"
#include "mso.h"
#include "ex4bit.h"

#define SOH     0x01
#define STX     0x02
#define EOT     0x04
#define ACK     0x06
#define NAK     0x15
#define CAN     0x18

// Monochorome 128x128 bitmap header
byte EEMEM BMP[62] = {
    0x42,0x4D,0x3E,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x3E,0x00,0x00,0x00,0x28,0x00,
    0x00,0x00,0x80,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x01,0x00,0x01,0x00,0x00,0x00,
    0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0x00 };

void Send(byte tx) {
    while ( !testbit(UCSR0A,UDRE0) ) ;  // wait until ready to transmit
    UDR0 = tx;
}

// Receive interrupt
ISR(USART0_RX_vect) {
    byte rx;
    rx=UDR0;
    lcd_write_command(MWRITE);
    lcd_write_data(rx);       // echo character on LCD
}

// Send data from LCD to PC via RS-232 using the XMODEM protocol
void SendBMP(void) {
    byte rx,data,Packet=1, n=0,i,j;
    byte low, high, tempP, tempD;
    unsigned int Address;
    CRC16_High=0;
    CRC16_Low=0;

    tempP = PORTB;
    tempD = DDRC;        // Set port as output

    setbit(PORTB, ADS931NOUT);  // Disable ADC output
    DDRC = 0xFF;                // Set PortC as output

    // OR second layer on to first layer
    // Only works in LCD GRAPHICS Mode!
    for(Address=0; Address<2048; Address++) {
        // Read byte from second layer
        low = (byte) (  (Address+2048) & 0x00ff);
        high = (byte) (( (Address+2048) & 0xff00) >> 8);
        lcd_write_command(CSRW);        // Set cursor address
        lcd_write_data(low);
        lcd_write_data(high);
        lcd_write_command(MREAD);       // Read data
        data=lcd_read_data();

        // Read byte from first layer and OR
        low = (byte) (  (Address) & 0x00ff);
        high = (byte) (( (Address) & 0xff00) >> 8);
        lcd_write_command(CSRW);        // Set cursor address
        lcd_write_data(low);
        lcd_write_data(high);
        lcd_write_command(MREAD);       // Read data
        data|=lcd_read_data();
    
        // Write back to fisrt layer
        lcd_write_command(CSRW);        // Set cursor address
        lcd_write_data(low);
        lcd_write_data(high);
        lcd_write_command(MWRITE);       // Read data
        lcd_write_data(data);
    }

    // First Block
    Send(SOH);
    Send(Packet);       // Send Packet number
    Send(255-Packet);   // Send Packet number 1's complement
    // Send BMP Header
    for(i=0; i<62; i++) {
        n++;
        data = (char)eeprom_read_byte(&BMP[i]);
        CRC16_Update(data);
        Send(data);
    }
    lcd_write_command(CSR_RIGHT);

    // Send LCD data
    for(i=127; i!=255; i--) {
        Address = (unsigned int)i<<4;
        low = (byte) (Address & 0x00ff);
        high = (byte) ((Address & 0xff00) >> 8);
        lcd_write_command(CSRW);        // Set cursor address
        lcd_write_data(low);
        lcd_write_data(high);
        lcd_write_command(MREAD);

        for(j=0; j<16; j++) {
            data=~lcd_read_data();
            CRC16_Update(data);
            Send(data);
            n++;
            if(n==128)  {   // end of 128byte block
                n=0;
                Send(CRC16_High);
                Send(CRC16_Low);
                // Wait for ACK
                while (!testbit(UCSR0A,RXC0)) delay_ms(1);
                rx = UDR0;
                if(rx!=ACK) return; // Error -> cancel transmission
                Packet++;
                Send(SOH);
                Send(Packet);
                Send(255-Packet);
                CRC16_High=0;
                CRC16_Low=0;
            }
        }
    }

    // End of last block
    for(; n<128; n++) { // send remainder of block
        data= 0x1A;     // pad with 0x1A which marks the end of file
        CRC16_Update(data);
        Send(data);
    }
    Send(CRC16_High);
    Send(CRC16_Low);

    while (!testbit(UCSR0A,RXC0)) delay_ms(1);
    rx = UDR0;
    if(rx!=ACK) return;

    Send(EOT);
    while (!testbit(UCSR0A,RXC0)) delay_ms(1);
    rx = UDR0;
    if(rx!=NAK) return;
    Send(EOT);
    while (!testbit(UCSR0A,RXC0)) delay_ms(1);
    rx = UDR0;
    if(rx!=ACK) return;
    lcd_write_command(CSR_DOWN);
    PORTB = tempP;
    DDRC = tempD;
}

void initUART(void) {
    //Negative(1);
    clrbit(PRR,PRUSART0);       // power up USART0
    UBRR0H = 0;
    UBRR0L = 62;                // 19.2MHz / 16 (UBRR + 1) = 19200 bps
    // Enable receiver and transmitter, enable RX interrupt
    UCSR0B = (1<<RXCIE0)|(1<<RXEN0)|(1<<TXEN0);
    UCSR0C = (3<<UCSZ00);       // 8 data bits, 1 stop bit, no parity
    SMCR = 0x60;    // Sleep mode = power down, sleep disabled
}

