/*****************************************************************************

MultiKitB: AVR Oscilloscope and Development Kit

Gabotronics C.A.
May 2009

Copyright 2009 Gabriel Anzziani

This program is distributed under the terms of the GNU General Public License 

www.gabotronics.com
email me at: gabriel@gabotronics.com

AVR Fuse settings:
BODLEVEL    4.3V
OCDEN       OFF
JTAGEN      OFF
SPIEN       ON
WDTON       OFF
EESAVE      ON
BOOTSZ      1024
BOOTRST     OFF
CKDIV       OFF
CKOUT       OFF
SUT_CKSEL   EXT CRYST 8MHz 16K CK + 65ms (Crystal is 19.2MHz)

Compiled with GCC, -O2 optimizations

*****************************************************************************/

#include <avr/io.h>
#include <util/delay_basic.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include "mygccdef.h"
#include "sed1335.h"
#include "mso.h"
#include "asmutil.h"
#include "ffft.h"
#include "uart.h"

// Fuse settings if using ELF file
FUSES = {
    .low = 0xFF,                                    // EXT CRYST 8MHz 16K CK + 65ms
    .high = (FUSE_SPIEN & FUSE_EESAVE),             // SPI enabled, Save EE
    .extended = (FUSE_BODLEVEL1 & FUSE_BODLEVEL0),  // BODLEVEL    4.3V
};

 // Keypad
byte keylevel[16] PROGMEM =
{ 247, 233, 221, 209, 199, 190, 178, 166, 160, 151, 136, 125, 122, 116, 107, 51 };

// Global variables
volatile byte key=0,oldkey=0;    // Keypad
volatile byte ROT1=0,ROT2=0,MAX1=255,MAX2=255;    // Encoder handler variables

// Global variables
complex_t  bfly[FFT_N];		/* FFT buffer & Data buffer */
byte HSData[136];
byte rate=0xFF;
byte gain=0xFF;

byte EEMEM EErate=5;        // 200uS/div
byte EEMEM EEgain=5;        // 1V/div
byte EEMEM EEOption = 0x20; // Display continuos lines between samples
byte EEMEM EEStatus = 0x01; // Set autotrigger
// Offset calibration table
signed char EEMEM offsets[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

byte OCR0Aval[12] PROGMEM =  { 5, 11, 23, 59, 119, 239, 74, 149, 239, 74, 149, 239};
byte OCR0Bval[12] PROGMEM =  { 2, 5,  11, 29,  59, 119, 36,  75, 119, 36,  75, 119};
byte TCCR0Bval[12] PROGMEM = { 9, 9,   9,  9,   9,   9, 10,  10,   9, 10,  10,   9};
byte OCR1AHval[6] PROGMEM =  { 0x1D, 0x3A, 0x0E, 0x24, 0x49, 0x92 };
byte OCR1ALval[6] PROGMEM =  { 0x4B, 0x97, 0xA5, 0x9E, 0x3D, 0x7B };

// Hamming window = 0.53836-0.46164*COS(2*3.14159256*n/(FFT_N-1))
byte Hamming[128] PROGMEM = {
    9,  9, 10, 10, 10, 11, 12, 13, 14, 15, 16, 18, 19, 21, 23, 25,
   27, 29, 31, 34, 36, 38, 41, 44, 46, 49, 52, 55, 58, 60, 63, 66,
   69, 72, 75, 78, 81, 84, 86, 89, 92, 95, 97,100,102,104,107,109,
  111,113,115,117,118,120,121,122,123,124,125,126,127,127,127,127,
  127,127,127,127,126,125,124,123,122,121,120,118,117,115,113,111,
  109,107,104,102,100, 97, 95, 92, 89, 86, 84, 81, 78, 75, 72, 69,
   66, 63, 60, 58, 55, 52, 49, 46, 44, 41, 38, 36, 34, 31, 29, 27,
   25, 23, 21, 19, 18, 16, 15, 14, 13, 12, 11, 10, 10, 10,  9,  9 };

// milivolts or volts per pixel, 5 decimal fixed point format
const long milivolts[8] = { 62500, 156250, 312500, 625000, 1562500, 3125, 6250, 15625 };

static const long freqval[18] = {
    // Kilo Hertz
    320000000,160000000,80000000,32000000,16000000,8000000,3200000,
    // Hertz
    1600000000,800000000,320000000,160000000,80000000,32000000,16000000,8000000,3200000,1600000,800000 };

static char unitkHz[] PROGMEM = "KHZ";   // kHz
static char unitHz[]  PROGMEM = "HZ ";   // Hz
static char clean[]   PROGMEM = "          ";

void ShowCursorV(void);
void ShowCursorH(void);

byte mark1V=32, mark2V=96, mark1H=96, mark2H=32;

void sound(byte tone) {
    byte i,j;
    for(j=80; j; j--) {
        for(i=tone; i; i--) {
            asm("nop");
            asm("nop");
        }
		setbit(PIND, BUZZER);   // Toggle pin
	}
	clrbit(PORTD, BUZZER);
}

// Negative voltage module control
void Negative(byte set) {
    TCCR2A = 0x00;              // stop
    TCCR2B = 0x00;
    ASSR  = 0x00;               // set async mode
    if(set) {                   // Enable negative voltage
        clrbit(PRR,PRTIM2);
        TCCR2A = 0x42;          // start, OC2A toggle, waveform = CTC
        TCCR2B = 0x01;          // clk/1, freq
        OCR2A = 2;
    }                           // Disable negative voltage
    else {
        setbit(PRR,PRTIM2);     // power down tmr2
    }
}

int main(void) {
    byte grid=1, triggervalue=128, oldtrigger=128;
    byte oldOCR0A, oldOCR0B, oldT0;
    byte oldm1V, oldm2V, oldm1H, oldm2H;
    byte i,j,index,max,min;
    byte average,vpp,*dp;
    signed char offset;
    byte Encoders, digselect=0;
    char *unitF;
 
    unsigned int sum;
    static char gaintxt[][5] PROGMEM = {
    "20fg", "50fg", " 0.1", " 0.2",             // 20m, 50m, 0.1, 0.2
    " 0.5", "   1", "   2", "   5",             // 0.5, 1,   2,   5
    };
    static char ratetxt[][5] PROGMEM = {
    "  5a", " 10a", " 20a", " 50a",             // 5u,   10u,  20u, 50u
    "100a", "200a", "500a", " 1fg",             // 100u, 200u, 500u, 1m
    " 2fg", " 5fg", "10fg", "20fg",             // 2m,   5m,   10m, 20m
    "50fg", " 0.1", " 0.2", " 0.5",             // 50m,  0.1,  0.2, 0.5
    "   1", "   2"                              // 1,    2
    };
    static char Vdiv[]    PROGMEM = "Vcde";        // V/div
    static char Sdiv[]    PROGMEM = "Scde";        // S/div
    static char Normal[]  PROGMEM = " NORMAL";
    static char Off[]     PROGMEM = " OFF";
    static char Single[]  PROGMEM = " SINGLE";
    static char Digital[] PROGMEM = "DIGITAL ";

    delay_ms(100);  // Debounce ON Switch
	DIDR0 = 0x01;   // Disable digital input buffers
    PORTA = 0x10;   // Disable ADS931
	DDRA =  0xFC;   // SEL0-SEL2 outputs, PA2 output, LED output
    PORTB = 0x08;   // Disable ADS931 output
	DDRB =  0x1F;   // PB5-PB7 inputs
    ACSR =  0x80;   // Disable comparator
	PORTC = 0xFF;
	DDRC =  0xFF;   // Data port - LCD and ADC
	PORTD = 0x3C;   // Pull ups on rotary encoders
	DDRD =  0xC2;
    ADMUX = 0x20;   // ADC left adjusted, select CH0
    ADCSRA= 0x86;   // Enable ADC, XTAL/64 frequency
    PCICR = 0x09;   // Enable interrupt on change for PCIE0 and PCIE3
    PCMSK3= 0x3C;   // Rotary encoder pins change interrupt mask
    PCMSK0= 0x02;   // Power button switch pin change interrupt mask

    WDTCSR |= (1<<WDCE) | (1<<WDE); // Enable change on watchdog settings
    WDTCSR = (1<<WDIE) | (1<<WDP1)/* | (1<<WDP0)*/;   // Enable watchdog interrupt with 125mS period
    
    setbit(PIND, BUZZER);   // Toggle pin
    // Initialize LCD
    lcd_init(TEXT);
    lcd_puts(PSTR("MULTIKITB V1.3"));
    setbit(PIND, BUZZER);   // Toggle pin

    while(testbit(PINA, ONBUTTON)) PORTA = 0x10; // Wait until switch is depressed
    PORTA = 0x14;                   //Keep power up with PA2
    delay_ms(20);
    PCIFR = 0x0F;   // Clear PCIE flags (bouncing on ON button)

    sei();          // enable global interrupts
    setbit(Status, trigger);

    SMCR = 0x00;    // Sleep mode = idle, sleep disabled
    CLKPR = 0x80;   // Change clock prescaler
    CLKPR = 0x00;   // Clock prescaler = 1 (Full AVR Speed)
    PRR = 0xFF;     // Power down all AVR peripherals    for(;;) {

    Negative(1);    // Activate negative voltage
    clrbit(PORTA,ADS931POW);  // Power up ADS931

    ROT1=7-(char)eeprom_read_byte(&EEgain); MAX1 = 7; // gain
    ROT2=17-(char)eeprom_read_byte(&EErate); MAX2 = 17;// rate
    Option = (char)eeprom_read_byte(&EEOption);
    Status = (char)eeprom_read_byte(&EEStatus);
    setbit(Status, update);

    Encoders = GAINRATE;

    dp=(byte *)bfly;        // Use FFT Buffer to store digital data too

    initUART(); // To enable screen dunmp to RS-232
    clrbit(UCSR0B,RXCIE0);

    lcd_init(GRAPHICS);
    lcd_write_command(CSR_DOWN);

    clrbit(PRR,PRTIM0); // Power up Timer 0 - ACD Clock
    clrbit(PRR,PRTIM1); // Power up Timer 1 - Time base
    TCCR1A = 0;
    TCCR0A = 0x23;      // Fast PWM, OC0B clear on compare, set at BOTTOM
    OCR0A = 239;        // 80kHz
    OCR0B = 119;        // 50% duty cycle
    TCCR0B = 0x09;      // clk/8

    for(;;) {
        setbit(WDTCSR,WDIE);     // enable watchdog timer
        // Speed up ADS931 clock for upcoming trigger detect
        oldT0 = TCCR0B;
        oldOCR0A = OCR0A;
        oldOCR0B = OCR0B;
        OCR0A = 5;
        OCR0B = 2;
        TCCR0B = 9;            // Set ADS931 at high speed for trigger detect
        if(rate<=11) {          // If sampling too fast, go to sleep to limit refresh rate
            setbit(SMCR,SE);    // because it won't look good on LCD
            sleep_cpu();
        }
        // Wait for trigger, check key input
        do {
            if( !testbit(Status, trigger) &&    // Trigger not set already
                !testbit(Status, autotrg) &&    // Not in autotrigger mode
                !testbit(Status, stop)) {       // Not in stop mode
                clrbit(WDTCSR,WDIE);    // disable watchdog timer during trigger check
                TCNT0=0;
                if(testbit(Status, digitrig)) {
                    digitrigup();
                }
                else {
                    if(testbit(Status, trigdir)) trigdown(triggervalue);  // falling edge
                    else trigup(triggervalue);                        // rising edge
                }
                setbit(WDTCSR,WDIE);
            }

            if(testbit(Status, update)) {
                clrbit(Status, update);
                setbit(PORTB,ADS931NOUT);       // Disable ADC output
                DDRC = 0xFF;        // Set port as output
                switch(Encoders) {
                    case GAINRATE:  // Encoders are controlling gain and rate
                        if(gain!=(7-ROT1) || rate!=(17-ROT2)) {
                            gain=7-ROT1; rate=17-ROT2;
//                          eeprom_write_byte(&EEgain, gain);
//                          eeprom_write_byte(&EErate, rate);
                            PORTA = _BV(POWERUP) | (gain<<5);
                            offset=(signed char)eeprom_read_byte(&offsets[gain]);
                            if(rate>=12) {
                                // Set Time base with Timer1
                                OCR1AH = (byte)pgm_read_byte_near(OCR1AHval+rate-12);
                                OCR1AL = (byte)pgm_read_byte_near(OCR1ALval+rate-12);
                                TCNT1 = 0;
                                // Set ADS931 clock
                                oldOCR0A = OCR0A = 149;        // 16kHz
                                oldOCR0B = OCR0B = 75;         // 50% duty cycle
                                oldT0=TCCR0B = 0x0A;      // clk/8
                            }
                            else {
                                oldOCR0A = OCR0A = (byte)pgm_read_byte_near(OCR0Aval+rate);
                                oldOCR0B = OCR0B = (byte)pgm_read_byte_near(OCR0Bval+rate);
                                oldT0=TCCR0B = (byte)pgm_read_byte_near(TCCR0Bval+rate);
                            }
                            if(rate<=13) TCCR1B = 0x0A;      // CTC, clk/8
                            else TCCR1B = 0x0B;      // CTC, clk/64
                            lcd_clear_graphics();
                            lcd_write_command(CSR_DOWN);
                        }
                    break;
                    case CURSORH:   // Encoders are controlling horizontal cursors
                        if(testbit(Option, cursorh)) {
                            mark1H = ROT1;
                            mark2H = ROT2;
                            for(i=1; i<=126; i+=4) { 
                                lcd_pixel(i,128+oldm1H,0);
                                lcd_pixel(i+2,128+oldm2H,0);
                            }
                            ShowCursorH();
                            oldm1H = mark1H;
                            oldm2H = mark2H;
                        }
                    break;                        
                    case CURSORV:   // Encoders are controlling vertical cursors
                        if(testbit(Option, cursorv)) {
                            mark1V = ROT1;
                            mark2V = ROT2;
                            for(i=1; i<=126; i+=4) { 
                                lcd_pixel(oldm1V,128+i,0);
                                lcd_pixel(oldm2V,130+i,0);
                            }
                            ShowCursorV();
                            oldm1V = mark1V;
                            oldm2V = mark2V;
                        }
                    break;
                    case TRIGGER:
                        if(ROT2 != (Status&3)) {
                            switch(ROT2) {
                                case 0: // Trigger Normal
                                    clrbit(Status, single); clrbit(Status, autotrg);
                                    tiny_printp(11,123, Normal);
                                break;
                                case 1: // Trigger Free
                                    clrbit(Status, single); setbit(Status, autotrg);
                                    tiny_printp(12,123, PSTR(" FREE"));
                                break;
                                case 2: // Single Normal
                                    index=0;
                                    setbit(Status, single); clrbit(Status, autotrg);
                                    tiny_printp(11,123, Single);
                                    lcd_goto(15,123);
                                    if(testbit(Status, trigdir)) put_2char(0,'h');
                                    else put_2char(0,'i');
                                    lcd_clear_graphics();
                                    lcd_write_command(CSR_DOWN);
                                break;
                                case 3: // Single Free
                                    index=0;
                                    setbit(Status, single); setbit(Status, autotrg);
                                    tiny_printp(11,123, Single);
                                    lcd_clear_graphics();
                                    lcd_write_command(CSR_DOWN);
                                break;
                            }
                            clrbit(Status, stop);
                            delay_ms(250);
                        }
                        lcd_goto(0,triggervalue>>1); put_2char(' ',' ');
                        if(!testbit(Status, autotrg)) {  // Not auto trigger
                            oldtrigger=oldtrigger>>1; // to compare with ROT1
                            if(oldtrigger<ROT1) clrbit(Status, trigdir);
                            else if(oldtrigger>ROT1) setbit(Status, trigdir);
                            oldtrigger = triggervalue = ROT1<<1;
                            lcd_goto(0,triggervalue>>1);
                            if(testbit(Status, trigdir)) put_2char(0,'h');
                            else put_2char(0,'i');
                        }
                    break;
                }
                switch(key) {
                    case K1:    // Set Encoders to Gain & Rate
                        Encoders = GAINRATE;
                        ROT1 = 7-gain; MAX1 = 7; // Maximum gain
                        ROT2 = 17-rate; MAX2 = 17; // Maximum sampling rate
                        tiny_printp(4,123, PSTR(" GAIN       RATE"));
                    break;
                    case K2:    // Set Encoders to H cursors
                        Encoders = CURSORH;
                        togglebit(Option, cursorh);
                        if(testbit(Option, cursorh))
                            tiny_printp(0,123, PSTR("CURSORS   H1        H2 "));
                        else tiny_printp(0,123, PSTR("H CURSORS OFF"));
                        ROT1 = mark1H;
                        ROT2 = mark2H;
                        MAX1 = 127;
                        MAX2 = 127;
                    break;
                    case K12:   // Persistent mode toggle
                        togglebit(Option, persistent);
                        if(testbit(Option, persistent)) tiny_printp(0,123, PSTR("PERSISTENT"));
                        else tiny_printp(0,123, Normal);
                        lcd_clear_graphics();
                    break;
                    case K3:    // Set Encoders to V cursors
                        Encoders = CURSORV;
                        togglebit(Option, cursorv);
                        if(testbit(Option, cursorv))
                            tiny_printp(0,123, PSTR("CURSORS   V1        V2 "));
                        else tiny_printp(0,123, PSTR("V CURSORS OFF"));
                        ROT1 = mark1V;
                        ROT2 = mark2V;
                        MAX1 = 127;
                        MAX2 = 127;
                    break;
                    case K13:   // Hide / unhide gain and rate settings
                        togglebit(Option, hideset);
                        tiny_printp(0,123, PSTR("SHOW"));
                    break;
                    case K23:   // Line / Dot mode toggle
                        togglebit(Option, line);
                        if(testbit(Option, line)) tiny_printp(3,123, PSTR("LINE"));
                        else tiny_printp(3,123, PSTR("DOTS"));
                        lcd_clear_graphics();
                    break;
                    case K4:    // Math display
                        Encoders = GAINRATE;
                        ROT1 = 7-gain; MAX1 = 7; // Maximum gain
                        ROT2 = 17-rate; MAX2 = 17; // Maximum sampling rate
                        togglebit(Option, math);
                        if(testbit(Option, math)) {
                            tiny_printp(9,123, PSTR("MATH"));
                        }
                        else tiny_printp(9,123, Off);
                    break;
                    case K24:   // Calibrate offset
                        tiny_printp(0,123, PSTR("ADJUST"));
                        DDRC = 0;           // Set port as input
                        clrbit(PORTB,ADS931NOUT);       // Enable ADC output
                        OCR0A = 149;        // 16kHz
                        OCR0B = 75;         // 50% duty cycle
                        TCCR0B = 0x0A;      // clk/8
                        for(i=0; i<8; i++) {
                            PORTA = _BV(POWERUP) | (i<<5);
                            delay_ms(50);
                            sum=0;
                            TCNT0 = 0; // Sync Timer0 with ADS931
                            for(j=0; j<128; j++) {
                                while(!testbit(TIFR0,OCF0B)) ;
                                sum+=PINC;
                                setbit(TIFR0,OCF0B);
                            }
                            average=sum/128;
                            if(average>=128) offset=-(average-128);
                            else offset = (128-average);
                            eeprom_write_byte(&offsets[i], offset);
                        }
                        PORTA = _BV(POWERUP) | (gain<<5);
                        offset=(signed char)eeprom_read_byte(&offsets[gain]);
                        setbit(PORTB, ADS931NOUT);   // Disable ADC output
                        DDRC = 0xFF;        // Set port as output
                    break;
                    case K34:   // Grid type
                        grid++;
                        if(grid>=4) grid=0;
                        tiny_printp(5,123, PSTR("GRID"));
                    break;
                    case K5:    // FFT Display
                        Encoders = CURSORV;
                        ROT1 = mark1V;
                        ROT2 = mark2V;
                        MAX1 = 127;
                        MAX2 = 127;
                        togglebit(Status, fft);
                        if(testbit(Status, fft)) {
                            tiny_printp(12,123, PSTR("FFT"));
                        }
                        else tiny_printp(11,123, Normal);
                        lcd_clear_graphics();
                        lcd_write_command(CSR_DOWN);
                    break;
//                    case K15:
//                    case K35:
//                    break;
                    case K45:
                        togglebit(Option, nowindow);
                        tiny_printp(0,123, PSTR("WINDOW"));
                    break;
                    case K6:    // Toggle Digital Inputs, set encoders to digital settings
                        Encoders = GAINRATE;
                        ROT1 = 7-gain; MAX1 = 7; // Maximum gain
                        ROT2 = 17-rate; MAX2 = 17; // Maximum sampling rate
                        tiny_printp(9,123, Digital);
                        if(testbit(Option, digital)) {
                            if(testbit(Status, digitrig)) {
                                clrbit(Option, digital);
                                tiny_printp(13,123, Off);
                            } else {
                                setbit(Status, digitrig);
                                tiny_printp(13,123, PSTR("TRG"));
                            }
                        }
                        else {
                            if(testbit(Status, digitrig)) {
                                clrbit(Status, digitrig);
                                tiny_printp(13,123, PSTR("TRGOFF"));
                            }
                            else setbit(Option, digital);
                        }
                    break;
                    case K7:    // Trigger button
                        if(Encoders!=TRIGGER) {
                            Encoders = TRIGGER;
                            tiny_printp(0,123, PSTR("TRIGGER  LEVEL      TYPE "));
                            delay_ms(250);
                        }
                        else {
                            // Trigger is Single -> Enable one more trace
                            if(testbit(Status, single)) {
                                index=0;
                                lcd_clear_graphics();
                                clrbit(Status, stop);
                            }
                            // Trigger is Normal or Free -> Toggle stop / run
                            else {
                                if(testbit(Status, stop)) {
                                    tiny_printp(14,123, PSTR("RUN"));
                                }
                                else {
                                    clrbit(Status, trigger);  // clear trigger
                                    tiny_printp(13,123, PSTR(" STOP"));
                                }
                                togglebit(Status, stop);
                                delay_ms(250);
                            }
                        }
                        ROT1 = oldtrigger>>1;
                        MAX1 = 127;
                        ROT2 = Status & 3;
                        MAX2 = 3;
                        key=NOKEY;  // special case for ON button
                    break;
                }
                while(key) {            // wait for key release
                    setbit(SMCR,SE);    // Enable Sleep
                    sleep_cpu();
                }
                // Update second layer graphics
                lcd_clear_graphics2();
                lcd_write_command(CSR_DOWN);
                // Grid
                switch(grid) {
                    case 1: for(i=16; i<=112; i+=16) { lcd_pixel(i,192,1); lcd_pixel(64,i+128,1); }
                        break;
                    case 2: lcd_line(0,192,127,192,1); lcd_line(64,128,64,255,1);
                            // mini grid for digital inputs
                            if(testbit(Option, digital)) {
                                for(j=72; j<=112; j+=8) {
                                    for(i=16; i<=112; i+=16) lcd_pixel(i,j+128,1);
                                }
                            }
                    break;
                    case 3: for(j=16; j<=112; j+=16) {
                                for(i=16; i<=112; i+=16) lcd_pixel(i,j+128,1);
                            }
                    break;
                }
                ShowCursorH(); 
                ShowCursorV();
                // Display time and gain settings
                if(!testbit(Option, hideset)) {
                    tiny_printp(12,0,gaintxt[gain]);
                    tiny_printp(14,0,Vdiv);    // V/div
                    tiny_printp(12,7,ratetxt[rate]);
                    tiny_printp(14,7,Sdiv);    // S/div
                }
                if(Option!=(char)eeprom_read_byte(&EEOption)) {
//                    eeprom_write_byte(&EEOption, Option);
                }
                // Only check bit in Status that should be saved
                if((Status&115)!=(char)eeprom_read_byte(&EEOption)) {
//                    eeprom_write_byte(&EEStatus, Status);
                }
                // Trigger mark
                if(!testbit(Status, autotrg)) {
                    lcd_goto(0,triggervalue>>1);
                    if(testbit(Status, trigdir)) put_2char(0,'h');
                    else put_2char(0,'i');
                }
                DDRC = 0;           // Set port as input
                clrbit(PORTB,ADS931NOUT);       // Enable ADC output
            }
        } while(!testbit(Status, trigger));

        TCCR0B = oldT0; // recover ADS931 clock speed after trigger check
        OCR0A = oldOCR0A;
        OCR0A = oldOCR0A;

        if(rate<12) clrbit(WDTCSR,WDIE);    // disable watchdog timer
        TCNT0 = 0; // Sync Timer0 with ADS931
        // Acquite data
        switch(rate) {
            case 0:     // 5uS   / div  ->  312.5nS / pixel ->  3.2 MS / s
                acquire(HSData, dp);    // acquire data fast
                clrbit(Status, trigger);
                break;
            case 1:     // 10uS  / div  ->  625nS   / pixel ->  1.6 MS / s
                acquire2(HSData, dp);    // acquire data
                clrbit(Status, trigger);
                break;
            case 2:     // 20us  / div  ->  1.25uS  / pixel ->  800 kS / s
            case 3:     // 50uS  / div  ->  3.125uS / pixel ->  320 kS / s
            case 4:     // 100uS / div  ->  6.25uS  / pixel ->  160 kS / s
            case 5:     // 200uS / div  ->  12.5uS  / pixel ->  80  kS / s
            case 6:     // 500uS / div  ->  31.25uS / pixel ->  32  kS / s
            case 7:     // 1mS   / div  ->  62.5uS  / pixel ->  16  kS / s
                acquire3(HSData, dp);
                clrbit(Status, trigger);
                for(i=0; i<128; i++) HSData[i]=HSData[i+8];    // Ignore first 8 samples

                break;
            case 8:     // 2mS   / div  ->  125uS   / pixel ->  8   kS / s
            case 9:     // 5mS   / div  ->  312.5uS / pixel ->  3.2 kS / s
            case 10:    // 10mS  / div  ->  625uS   / pixel ->  1.6 kS / s
                TCNT0 = 0;  // Sync Timer0 with ADS931
                setbit(TIFR0,OCF0B); // clear flag (counter intuitive)

                for(i=0; i<134; i++) {
                    sum=0;
                    for(j=0; j<10; j++) {
                        while(!testbit(TIFR0,OCF0B)) ;
                        sum+=PINC;
                        setbit(TIFR0,OCF0B);
                    }
                    HSData[i]=sum/10;
                    dp[i]=PINB;
                }
                for(i=0; i<128; i++) HSData[i]=HSData[i+6];    // Ignore first 6 samples
                clrbit(Status, trigger);
                break;
            case 11:    // 20mS  / div  ->  1.25mS  / pixel ->  800  S / s
                TCNT0 = 0;  // Sync Timer0 with ADS931
                setbit(TIFR0,OCF0B); // clear flag (counter intuitive)

                for(i=0; i<128; i++) {
                    sum=0;
                    for(j=0; j<100; j++) {
                        while(!testbit(TIFR0,OCF0B)) ;
                        sum+=PINC;
                        setbit(TIFR0,OCF0B);
                    }
                    HSData[i]=sum/100;
                    dp[i]=PINB;
                }
                clrbit(Status, trigger);
                break;
            case 12:    // 50mS  / div  ->  3.125mS / pixel ->  320  S / s
            case 13:    // 100mS / div  ->  6.25mS  / pixel ->  160  S / s
            case 14:    // 200mS / div  ->  12.5mS  / pixel ->  80   S / s
            case 15:    // 500mS / div  ->  31.25mS / pixel ->  32   S / s
            case 16:    // 1S    / div  ->  62.5mS  / pixel ->  16   S / s
            case 17:    // 2S    / div  ->  125mS   / pixel ->  8    S / s
                sum=0;

                delay_ms(20);

                break;
        }

        setbit(PORTB, ADS931NOUT);   // Disable ADC output
        DDRC = 0xFF;        // Set port as output

        // Frequency analisys
        if(testbit(Status, fft) && !testbit(Status, trigger)) {
            for(i=FFT_N-1; i!=255; i--) {
                HSData[i]=(signed char)(HSData[i]-128); // Convert to signed char
                if(!testbit(Option, nowindow)) {
                    HSData[i]=fmuls_8(HSData[i],pgm_read_byte_near(Hamming+i)); // Apply Hamming window
                }
                bfly[i].r=bfly[i].i=(signed int)HSData[i]*256;
            }
            fft_execute(bfly);
            fft_output(bfly, HSData);

            // Index reverse
            for (i=1,j=0;i<(FFT_N/2)-1;i++) {
                max=(FFT_N/2)/2;
                while (j >= max) {
                    j -= max;
                    max = max/2;
                }
                j += max;
                if (j > i) SWAP(HSData[j],HSData[i]);
            }

            // Erase old data
            if(!testbit(Option, persistent)) {
                lcd_clear_graphics();
                lcd_write_command(CSR_DOWN);
            }
            // Display new data
            for(i=0,j=0; j<64; i+=2,j++) {
                if(testbit(Option, line)) {
                    lcd_line(i, 127-(HSData[j]>>1), i, 127, 1);
                    lcd_line(i+1, 127-(HSData[j]>>1), i+1, 127, 1);
                }
                else {
                    lcd_pixel(i, 127-(HSData[j]>>1),1);
                    lcd_pixel(i+1, 127-(HSData[j]>>1),1);
                }
            }
            setbit(PORTA, LED);
        }
        else if(rate<=11) {
            // Erase old data
            if(!testbit(Option, persistent)) {
                lcd_clear_graphics();
                lcd_write_command(CSR_DOWN);
            }
            // Adjust offset
            for(i=0; i<128; i++) {
                if(offset>0) {
                    if(HSData[i]>255-offset) HSData[i]=255;
                    else HSData[i]+=offset;
                }
                else if(offset<0) {
                    if(HSData[i]<(-offset)) HSData[i]=0;
                    else HSData[i]+=offset;
                }
            }
            // Display new data
            for(i=0; i<128; i++) {
                if(testbit(Option, digital)) {  // Digital Data
                    if(testbit(dp[i],PB5)) lcd_pixel(i, 72, 1); else { lcd_pixel(i, 80, 1); lcd_pixel(i, 81, 1); }
                    if(testbit(dp[i],PB6)) lcd_pixel(i, 88, 1); else { lcd_pixel(i, 96, 1); lcd_pixel(i, 97, 1); }
                    if(testbit(dp[i],PB7)) lcd_pixel(i, 104, 1); else { lcd_pixel(i, 112, 1); lcd_pixel(i, 113, 1); }
                }
                if(testbit(Option, line)) {
                    if(i==127) continue;
                    else lcd_line(i, HSData[i]>>1, i+1, HSData[i+1]>>1, 1);
                }
                else lcd_pixel(i, HSData[i]>>1,1);
            }
            setbit(PORTA, LED);
        }
        else if (!testbit(Status, fft)) {  // Slow sampling rate: clear next vertical line
            if(!testbit(Option, persistent)) {
                if(index==127) {
                    setbit(PORTA, LED);
                    lcd_line(0,0,0,127,0);
                }
                else lcd_line(index+1,0,index+1,127,0);
            }
            else if(index==127) { setbit(PORTA, LED); }
        }


        if(testbit(Option,math) && !testbit(Status, trigger)) {
            if(testbit(Status, fft)) {  // Math in FFT mode
                max=HSData[1]; vpp=1;   // Ignore DC
                for(i=2; i<64; i++) {   // Find max, vpp will hold the index
                    if(HSData[i]>max) {
                        max=HSData[i]; vpp=i;
                    }
                }
                if(rate<=6) unitF = unitkHz;
                else unitF = unitHz;
                if(max>1) { // Significant amplitude
                    // vpp contains the index of the max freq
                    printF(11,14,(long)(vpp)*(freqval[rate]/128));
                    tiny_printp(14,14,unitF);
                }
                else tiny_printp(11,14,clean);
            }
            else {  // Math in Normal mode
                sum=0;
                max=HSData[0]; min=HSData[0];
                for(i=0; i<128; i++) {
                    if(HSData[i]>max) max=HSData[i];
                    if(HSData[i]<min) min=HSData[i];
                    sum+=HSData[i];
                }
                vpp = max-min;
                average=sum/128;     // Average
                printF(11,14,(128-(long)average)*milivolts[gain]);
                printF(11,21,((long)vpp)*milivolts[gain]);
            }
        }
        else _delay_loop_2(2000);   // Makes LED stay on for a little bit
        clrbit(PORTA, LED);
        if(testbit(UCSR0A,RXC0)) {
            i=UDR0;
            if(i=='C') SendBMP();
        }
        DDRC = 0;           // Set port as input
        clrbit(PORTB,ADS931NOUT);       // Enable ADC output
        // If single trigger, now go to stop
        if(testbit(Status, single)) setbit(Status, stop);
    }
    return 0;
}

// Display Vertical Cursor
void ShowCursorV(void) {

}

// Display Horizontal Cursor
void ShowCursorH(void) {
    static char unitV[] PROGMEM = "V  ";    // V
    static char unitmV[] PROGMEM = "fgV";   // mV
    static byte oldmark1Hy=0, oldmark2Hy=0;
    char *markunit;
    byte i,y;

    if(gain<5) markunit = unitmV;
    else markunit = unitV;

    if(testbit(Option, math)) {
        if(testbit(Status, fft)) {
            tiny_printp(10,14, PSTR("F="));       // Fundamental F
        }
        else {
            tiny_printp(9,14, PSTR(" DC="));    // Average
            tiny_printp(14,14,markunit);
            tiny_printp(9,21, PSTR(" PP="));    // Peak to peak
            tiny_printp(14,21,markunit);
        }
    }
}

// Watchdog interrupt, 64mS period
// This interrupt is used among all applications for user input
ISR(WDT_vect) {
    static byte oldin=0;
    byte i;
    clrbit(PRR,PRADC);              // Power up ADC
    setbit(ADCSRA, ADSC);	        // Start conversion
    clrbit(SMCR,SE);                // Disable Sleep
    // Trigger for FREE mode:
    if(testbit(Status, autotrg) && !testbit(Status, stop)) setbit(Status, trigger);
	while(testbit(ADCSRA,ADSC));    // wait
	// Read Buttons
	if(oldin!=ADCH) {
		for(i=16; i>0; i--) {			    // DeBounce
       		setbit(ADCSRA, ADSC);		    // Start conversion (key input)
       		while(testbit(ADCSRA,ADSC));    // wait
			if(oldin!=ADCH) { oldin=ADCH; i++; }
            oldkey = key;
            setbit(Status, update);         // Valid key
		}
		for(key=0; key<16; key++) {
            if(ADCH>(byte)pgm_read_byte_near(keylevel+key)) break;
        }
    }
    setbit(PRR,PRADC);  // Power down ADC
    if(testbit(UCSR0A,RXC0)) {
       i=UDR0;
       if(i=='C') SendBMP();    // XMODEM CRC file request
    }
}

// POWER button
ISR(PCINT0_vect) {
    byte i=0,j;
    delay_ms(10);
    oldkey = key;
    if(testbit(PINA, ONBUTTON)) {
        setbit(Status, update);         // Valid key
        key = K7;
    }
    while(testbit(PINA, ONBUTTON)) {
        delay_ms(10);
        i++;
        if(i==125) {
            cli();  // disable interrupts
            // Shut down sound
            for(i=0; i<40; i++) {
		        for(j=i; j; j--) {
                    delay_ms(1);
                }
		        setbit(PIND, BUZZER);   // Toggle pin
	        }
            for(;;) PORTA = 0x10;  // TURN OFF SYSTEM!!!
        }
    }
}

// Rotary Encoders
//                _______    
// OutA:  _______|       |_____
//            _______     
// OutB: ____|       |______
//                      
// Data:   0   1   3   2
//
ISR(PCINT3_vect) {
    static byte oldEnc1=0,oldEnc2=0, Decision1=0, Decision2=0;
    byte Enc1, Enc2, Go1=0, Go2=0;

    Enc1 = (PIND & 0x0C) >> 2;
    Enc2 = (PIND & 0x30) >> 4;
    // Check Encoder 1
    if((Enc1==0) && (Decision1==0)) {  // Encoder is 00
        if(oldEnc1==1) Go1=2; // Increment
        if(oldEnc1==2) Go1=1; // Decrement
        Decision1=1;
    }
    else if((Enc1==3) && (Decision1==1)) {  // Encoder is 11
        if(oldEnc1==2) Go1=2; // Increment
        if(oldEnc1==1) Go1=1; // Decrement
        Decision1=0;
    }
    // Check Encoder 2
    if((Enc2==0) && (Decision2==0)) {  // Encoder is 00
        if(oldEnc2==1) Go2=2; // Increment
        if(oldEnc2==2) Go2=1; // Decrement
        Decision2=1;
    }
    else if((Enc2==3) && (Decision2==1)) {  // Encoder is 11
        if(oldEnc2==2) Go2=2; // Increment
        if(oldEnc2==1) Go2=1; // Decrement
        Decision2=0;
    }

    oldEnc1 = Enc1;
    oldEnc2 = Enc2;

    if(Go1==1) {
        if(ROT1<MAX1) { ROT1++; sound(50); }
        else { sound(250); ROT1=0; }  // Roll over
    }
    if(Go1==2) {
        if(ROT1>0) { ROT1--; sound(50); }
        else { sound(250); ROT1=MAX1; } // Roll over
    }

    if(Go2==1) {
        if(ROT2<MAX2) { ROT2++; sound(50); }
        else { sound(250); ROT2=0; } // Roll over
    }
    if(Go2==2) {
        if(ROT2>0) { ROT2--; sound(50); }
        else { sound(250); ROT2=MAX2; } // Roll over
    }

    if(Go1 || Go2) setbit(Status, update);
}
